#ifndef __NET_VXLAN_WRAPPER_H
#define __NET_VXLAN_WRAPPER_H 1

#ifdef CONFIG_INET
#include <net/udp_tunnel.h>
#endif


#ifdef HAVE_METADATA_DST
#include_next <net/vxlan.h>

static inline int rpl_vxlan_init_module(void)
{
	return 0;
}
static inline void rpl_vxlan_cleanup_module(void)
{}

#define vxlan_xmit dev_queue_xmit

#else

#include <linux/ip.h>
#include <linux/ipv6.h>
#include <linux/if_vlan.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/udp.h>
#include <net/dst_metadata.h>

#include "compat.h"
#include "gso.h"

#define VNI_HASH_BITS	10
#define VNI_HASH_SIZE	(1<<VNI_HASH_BITS)

/*
 * VXLAN Group Based Policy Extension:
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |1|-|-|-|1|-|-|-|R|D|R|R|A|R|R|R|        Group Policy ID        |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                VXLAN Network Identifier (VNI) |   Reserved    |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *
 * D = Don't Learn bit. When set, this bit indicates that the egress
 *     VTEP MUST NOT learn the source address of the encapsulated frame.
 *
 * A = Indicates that the group policy has already been applied to
 *     this packet. Policies MUST NOT be applied by devices when the
 *     A bit is set.
 *
 * [0] https://tools.ietf.org/html/draft-smith-vxlan-group-policy
 */
struct vxlanhdr_gbp {
	__u8	vx_flags;
#ifdef __LITTLE_ENDIAN_BITFIELD
	__u8	reserved_flags1:3,
		policy_applied:1,
		reserved_flags2:2,
		dont_learn:1,
		reserved_flags3:1;
#elif defined(__BIG_ENDIAN_BITFIELD)
	__u8	reserved_flags1:1,
		dont_learn:1,
		reserved_flags2:2,
		policy_applied:1,
		reserved_flags3:3;
#else
#error	"Please fix <asm/byteorder.h>"
#endif
	__be16	policy_id;
	__be32	vx_vni;
};

#define VXLAN_GBP_USED_BITS (VXLAN_HF_GBP | 0xFFFFFF)

/* skb->mark mapping
 *
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |R|R|R|R|R|R|R|R|R|D|R|R|A|R|R|R|        Group Policy ID        |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
#define VXLAN_GBP_DONT_LEARN		(BIT(6) << 16)
#define VXLAN_GBP_POLICY_APPLIED	(BIT(3) << 16)
#define VXLAN_GBP_ID_MASK		(0xFFFF)

#ifndef __VXLAN_GPE_HEADER
#define __VXLAN_GPE_HEADER 1
/*
 * VXLAN Generic Protocol Extension:
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |R|R|Ver|I|P|R|O|R|R|R|R|R|R|R|R|R|R|R|R|R|R|R|R|  Next Proto   |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                VXLAN Network Identifier (VNI) |   Reserved    |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * Ver = Version. Indicates VXLAN GPE protocol version.  The initial
 *      version is 0.  If a receiver does not support the version
 *      indicated it MUST drop the packet.
 *
 * I = Instance Bit. The I bit MUST be set to indicate a valid VNI.
 *
 * P = Next Protocol Bit. The P bit is set to indicate that the
 *     Next Protocol field is present.
 *
 * O = OAM Flag Bit. The O bit is set to indicate that the packet
 *     is an OAM packet.
 *
 * Next Protocol = This 8 bit field indicates the protocol header
 *     immediately following the VXLAN GPE header.
 *
 * [1] https://www.ietf.org/id/draft-ietf-nvo3-vxlan-gpe-01.txt
 */

struct vxlanhdr_gpe {
#ifdef __LITTLE_ENDIAN_BITFIELD
       uint8_t    oam_flag:1;
       uint8_t    reserved_flags1:1;
       uint8_t    np_applied:1;
       uint8_t    instance_applied:1;
       uint8_t    gpe_version:2;
       uint8_t    reserved_flags2:2;
#elif defined(__BIG_ENDIAN_BITFIELD)
       uint8_t    reserved_flags2:2;
       uint8_t    gpe_version:2;
       uint8_t    instance_applied:1;
       uint8_t    np_applied:1;
       uint8_t    reserved_flags1:1;
       uint8_t    oam_flag:1;
#else
#error "Please fix <asm/byteorder.h>"
#endif
    uint8_t    reserved_flags3;
    uint8_t    reserved_flags4;
    uint8_t    next_proto;
    __be32        vx_vni;
};

/* VxLAN-GPE Header Next Protocol */
#define VXLAN_GPE_NP_IPV4        0x01
#define VXLAN_GPE_NP_IPV6        0x02
#define VXLAN_GPE_NP_ETHERNET    0x03
#define VXLAN_GPE_NP_NSH        0x04

/* skb->mark mapping
 *
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |R|R|Ver|I|P|R|O|R|R|R|R|R|R|R|R|R|R|R|R|R|R|R|R|  Next Proto   |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */

#define VXLAN_GPE_OAM_FLAG                     (BIT(0) << 24)
#define VXLAN_GPE_NP_APPLIED           (BIT(0) << 26)
#define VXLAN_GPE_INSTANCE_APPLIED     (BIT(0) << 27)
#define VXLAN_GPE_VERSION                      ((BIT(0) << 28) | (BIT(0) << 29))

#define VXLAN_GPE_NP_MASK                      (0xFF)

#define VXLAN_GPE_USED_BITS (VXLAN_GPE_OAM_FLAG | VXLAN_GPE_NP_APPLIED \
                           | VXLAN_GPE_INSTANCE_APPLIED | VXLAN_GPE_VERSION | 0xFF)

#define VXLAN_HF_GPE BIT(26)
#define VXLAN_F_GPE                     0x4000

#endif /* __VXLAN_GPE_HEADER */

/* VXLAN protocol header:
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |G|R|R|R|I|R|R|C|               Reserved                        |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |                VXLAN Network Identifier (VNI) |   Reserved    |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *
 * G = 1	Group Policy (VXLAN-GBP)
 * I = 1	VXLAN Network Identifier (VNI) present
 * C = 1	Remote checksum offload (RCO)
 */
struct vxlanhdr {
	__be32 vx_flags;
	__be32 vx_vni;
};

/* VXLAN header flags. */
#define VXLAN_HF_RCO BIT(21)
#define VXLAN_HF_VNI BIT(27)
#define VXLAN_HF_GBP BIT(31)

/* Remote checksum offload header option */
#define VXLAN_RCO_MASK  0x7f    /* Last byte of vni field */
#define VXLAN_RCO_UDP   0x80    /* Indicate UDP RCO (TCP when not set *) */
#define VXLAN_RCO_SHIFT 1       /* Left shift of start */
#define VXLAN_RCO_SHIFT_MASK ((1 << VXLAN_RCO_SHIFT) - 1)
#define VXLAN_MAX_REMCSUM_START (VXLAN_RCO_MASK << VXLAN_RCO_SHIFT)

#define VXLAN_N_VID     (1u << 24)
#define VXLAN_VID_MASK  (VXLAN_N_VID - 1)
#define VXLAN_VNI_MASK  (VXLAN_VID_MASK << 8)
#define VXLAN_HLEN (sizeof(struct udphdr) + sizeof(struct vxlanhdr))

struct vxlan_metadata {
	__be32          vni;
	u32             gbp;
	u32             gpe;
};

#define VNI_HASH_BITS	10
#define VNI_HASH_SIZE	(1<<VNI_HASH_BITS)
#define FDB_HASH_BITS	8
#define FDB_HASH_SIZE	(1<<FDB_HASH_BITS)

/* per UDP socket information */
struct vxlan_sock {
	struct hlist_node hlist;
	struct work_struct del_work;
	struct socket	 *sock;
	struct rcu_head	  rcu;
	struct hlist_head vni_list[VNI_HASH_SIZE];
	atomic_t	  refcnt;
#ifdef HAVE_UDP_OFFLOAD
	struct udp_offload udp_offloads;
#endif
	u32		  flags;
};

union vxlan_addr {
	struct sockaddr_in sin;
	struct sockaddr_in6 sin6;
	struct sockaddr sa;
};

struct vxlan_rdst {
	union vxlan_addr	 remote_ip;
	__be16			 remote_port;
	u32			 remote_vni;
	u32			 remote_ifindex;
	struct list_head	 list;
	struct rcu_head		 rcu;
};

struct vxlan_config {
	union vxlan_addr	remote_ip;
	union vxlan_addr	saddr;
	u32			vni;
	int			remote_ifindex;
	int			mtu;
	__be16			dst_port;
	__u16			port_min;
	__u16			port_max;
	__u8			tos;
	__u8			ttl;
	u32			flags;
	unsigned long		age_interval;
	unsigned int		addrmax;
	bool			no_share;
};

/* Pseudo network device */
struct vxlan_dev {
	struct hlist_node hlist;	/* vni hash table */
	struct list_head  next;		/* vxlan's per namespace list */
	struct vxlan_sock *vn_sock;	/* listening socket */
	struct net_device *dev;
	struct net	  *net;		/* netns for packet i/o */
	struct vxlan_rdst default_dst;	/* default destination */
	u32		  flags;	/* VXLAN_F_* in vxlan.h */

	struct timer_list age_timer;
	spinlock_t	  hash_lock;
	unsigned int	  addrcnt;

	struct vxlan_config	cfg;
	struct hlist_head fdb_head[FDB_HASH_SIZE];
};

#define VXLAN_F_LEARN			0x01
#define VXLAN_F_PROXY			0x02
#define VXLAN_F_RSC			0x04
#define VXLAN_F_L2MISS			0x08
#define VXLAN_F_L3MISS			0x10
#define VXLAN_F_IPV6			0x20
#define VXLAN_F_UDP_CSUM		0x40
#define VXLAN_F_UDP_ZERO_CSUM6_TX	0x80
#define VXLAN_F_UDP_ZERO_CSUM6_RX	0x100
#define VXLAN_F_REMCSUM_TX		0x200
#define VXLAN_F_REMCSUM_RX		0x400
#define VXLAN_F_GBP			0x800
#define VXLAN_F_REMCSUM_NOPARTIAL	0x1000
#define VXLAN_F_COLLECT_METADATA	0x2000

/* Flags that are used in the receive path. These flags must match in
 * order for a socket to be shareable
 */
#define VXLAN_F_RCV_FLAGS		(VXLAN_F_GBP |			\
                                         VXLAN_F_GPE |                  \
					 VXLAN_F_UDP_ZERO_CSUM6_RX |	\
					 VXLAN_F_REMCSUM_RX |		\
					 VXLAN_F_REMCSUM_NOPARTIAL |	\
					 VXLAN_F_COLLECT_METADATA)
#define vxlan_dev_create rpl_vxlan_dev_create
struct net_device *rpl_vxlan_dev_create(struct net *net, const char *name,
				    u8 name_assign_type, struct vxlan_config *conf);

static inline __be16 vxlan_dev_dst_port(struct vxlan_dev *vxlan)
{
	return inet_sk(vxlan->vn_sock->sock->sk)->inet_sport;
}

static inline netdev_features_t vxlan_features_check(struct sk_buff *skb,
						     netdev_features_t features)
{
	u8 l4_hdr = 0;

	if (!skb->encapsulation)
		return features;

	switch (vlan_get_protocol(skb)) {
	case htons(ETH_P_IP):
		l4_hdr = ip_hdr(skb)->protocol;
		break;
	case htons(ETH_P_IPV6):
		l4_hdr = ipv6_hdr(skb)->nexthdr;
		break;
	default:
		return features;
	}

	if ((l4_hdr == IPPROTO_UDP) && (
#ifdef ENCAP_TYPE_ETHER
	    skb->inner_protocol_type != ENCAP_TYPE_ETHER ||
#endif
	     ovs_skb_get_inner_protocol(skb) != htons(ETH_P_TEB) ||
	     (skb_inner_mac_header(skb) - skb_transport_header(skb) !=
	      sizeof(struct udphdr) + sizeof(struct vxlanhdr))))
		return features & ~(NETIF_F_ALL_CSUM | NETIF_F_GSO_MASK);

	return features;
}

/* IP header + UDP + VXLAN + Ethernet header */
#define VXLAN_HEADROOM (20 + 8 + 8 + 14)
/* IPv6 header + UDP + VXLAN + Ethernet header */
#define VXLAN6_HEADROOM (40 + 8 + 8 + 14)

static inline unsigned short vxlan_get_sk_family(struct vxlan_sock *vs)
{
	return vs->sock->sk->sk_family;
}

int rpl_vxlan_init_module(void);
void rpl_vxlan_cleanup_module(void);

#define vxlan_xmit rpl_vxlan_xmit
netdev_tx_t rpl_vxlan_xmit(struct sk_buff *skb);
#endif

#define vxlan_init_module rpl_vxlan_init_module
#define vxlan_cleanup_module rpl_vxlan_cleanup_module

#endif
